// (C) Copyright 2011-2013 Hewlett-Packard Development Company, L.P.
define([ 'hp/services/REST', 'jquery' ],

function(REST) { "use strict";

    return {
        /**
         * Creates a user.
         *
         * @param {Object} user object
         * @params {Object} handlers with error and success.
         */
        postUser : function(user, handlers) {
            REST.postObject('/rest/users', user,
                    handlers);
        },
        /**
         * Assigns User to Roles.
         *
         * @param {Object} roles object
         * @params {Object} handlers with error and success.
         */
        rolesUser : function(roles, handlers) {
            REST.putObject('/rest/users/role', roles, null, handlers);
        },
        /**
         * Get a user by name
         *
         * @param {String} uri of a user
         * @params {Object} handlers with error and success.
         */
         getUserByName: function(username, handlers) {
             REST.getURI('/rest/users/' + username, handlers);
         },
         /**
          * Get a user by uri
          *
          * @param {String} uri of a user
          * @params {Object} handlers with error and success.
          */
          getUserByUri: function(uri, handlers) {
              REST.getURI(uri, handlers);
          },
        /**
         * Get all roles for a user.
         *
         * @param {String} uri of a user
         * @params {Object} handlers with error and success.
         */
        getAllUserRoles : function(uri, handlers) {
            REST.getURI('/rest/users/role/' + uri, handlers);
        },
        /**
         * @public
         * Get the list of roles defined on the appliance
         * @param {Object} handlers Should contain a "success"
         *                 function and an "error" function. e.g.
         *                 { success: function(roles) {...},
         *                   error: function(errorInfo) {...}
         *                 }
         */
        getRoles : function (handlers) {
            REST.getURI('/rest/roles', handlers);
        },
        
        /**
         * @public
         * Get the list of groups defined on the appliance
         * @param {String} uname username of the user
         * @param {String} password of the user
         * @param {String} authDomain directory server chosen by the user
         * @param {Object} handlers with error and success.
         */
        getGroupList : function (uname, password, authDomain, handlers) {
            var loginInfoDto = {
                    userName : uname,
                    password : password,
                    authLoginDomain : authDomain
                };
                REST.postObject(
                          '/rest/logindomains/groups',loginInfoDto, handlers);
        },
        /**
         * Modifies a user.
         *
         * @param {Object} user object
         * @params {Object} handlers with error and success.
         */
        modifyUser : function(user, handlers) {
            REST.putObject('/security-base/rest/user', user, null,
                    handlers);
        },

        /**
         * Get logged in user rest call
         */
         usersLoggedIn : function(handlers){
             REST.getURI('/rest/active-user-sessions', handlers);
         },

        /**
         * Deletes a specified local user with user name
         *
         * @param {String} the name of the user
         * @param {Object} handlers with error and success.
         */
        deleteUser: function(userName, handlers){
            REST.deleteURI('/security-base/rest/user/'+userName,handlers);
        },
        
        /**
         * Deletes a specified Directory Group
         *
         * @param {String} uri of the directory group to role mapping
         * @param {Object} handlers with error and success.
         */
        deleteDirectoryGroup: function(uri, handlers){
            REST.deleteURI(uri,handlers);
        },

        /**
         * check if the provided username exists already
         *
         * @param {String} the name of the user
         * @param {Object} handlers with error and success.
         */
        checkLoginNameExists: function(userName, handlers){
            REST.postObject('/rest/users/validateLoginName/'+userName, null, handlers);
        },

        /**
         * Gets list of Active Directories configured in the appliance
         */
        activeDirectoryList: function(handlers) {
            var activeDirectoryConfiguredUri = '/rest/logindomains/global-settings';
            REST.getURI(activeDirectoryConfiguredUri, handlers);
        },
        /**
         * @public Get full details on a provider
         * @param {string}
         *            URI is use to get full details on the provider
         * @param {Object}
         *            handlers Must contain a "success" function and
         *            an "error" function. e.g.
         */
         getResource : function(uri, handlers) {
            REST.getURI(uri, handlers);
        },
        /**
         * @public updateGroupToRoleMappings 
         * @param {Object}
         *            group2rolemappingsConfig object with details to be
         *            created
         * @param {Object}
         *            handlers Must contain a "success" function and
         *            an "error" function. e.g.
         */
        updateGroupToRoleMappings : function(group2rolemappingsConfig,handlers) {
              var group2RolesMappingPerGroupValidationDto  = {
              "type":"Group2RolesMappingPerGroupValidationDto",
              "group2rolesPerGroup":{
                   "type":"Group2RolesMappingPerGroupDto",
                   "loginDomain":group2rolemappingsConfig.loginDomain,
                   "egroup" : group2rolemappingsConfig.egroup,
                   "roles" :  group2rolemappingsConfig.roles
                    },
                   "credentials" : {
                   "userName" :  group2rolemappingsConfig.userName,
                   "password" :  group2rolemappingsConfig.password
                    }
                };
             REST.putObject('/rest/logindomains/grouptorolemapping', group2RolesMappingPerGroupValidationDto,null, handlers);
          },

          getProductConfig : function(handlers) {
              REST.getURI('/rest/appliance/product-config', handlers);
          }

    };
});
