// (C) Copyright 2012-2014 Hewlett-Packard Development Company, L.P.
/**
 * @type {Style}
 */
define(['hp/core/Localizer',
    'jquery',
    'lib/jquery.dateFormat',
    'hp/lib/date'],
function(localizer) {"use strict";

    var Style = ( function() {
      
        // https://h10014.www1.hp.com/hpweb/experience/make/infographics/color.aspx
        var BLUES = ['#0096d6', '#59bbe4', '#99d5ef', '#d9eff9'];
        var GRAYS = ['#000000', '#87898b', '#b9b8bb', '#e5e8e8'];
        //var GREENS = ['#008b2b', '#40a860', '#7fc595', '#bfe2ca'];
        var PURPLES = ['#822980', '#8e3e8c', '#c094bf', '#e0c9df', '#f2e9f2'];
        
        var GRAPH_COLORS = [BLUES[0], PURPLES[2], GRAYS[1], BLUES[2], PURPLES[1]];
        var GRAPH_COLORS_FADED = [BLUES[1], PURPLES[3], GRAYS[2], BLUES[3], PURPLES[2]];
        var SUMMARY_COLORS = [BLUES[0], BLUES[1], BLUES[2]];
        var SUMMARY_BASE_COLOR = GRAYS[2];
        var STATUS_COLORS = {error: '#dc442f', warning: '#f4b51e', ok: '#42a425',
            unknown: '#848484', disabled: '#848484'};
        var MAP = {
            // If we get more than this in a row, collapse into a summary item.
            summaryThreshold: 20,
            // How far inside the item edge to anchor connection lines
            anchorOffset: 5,
            // How long to wait between setting the highlight classes and
            // redrawing the connection lines. We delay this to avoid too
            // much flickering when the user explores by moving the mouse around.
            redrawDelay: 100,
            // Connection line colors
            selected: {color: '#7fbfda', width: 3,
                subduedColor: '#d9ecf4'}, //'rgba(127, 191, 218, 0.3)'},
            special: {color: PURPLES[1], width: 1},
            primary: {color: GRAYS[2], width: 1},
            highlight: {color: GRAYS[0], width: 3}
        };
        
        var DASHBOARD_BAR_GRAPH_OPTIONS = {
            series: { bars: { show: true, lineWidth: 0, fill: 1 } },
            grid: { show: false },
            legend: { show: false }
        };
        
        var DASHBOARD_LINE_GRAPH_OPTIONS = {
            series: { lines: { show: true, lineWidth: 4, fill: false } },
            grid: {borderWidth: 0, aboveDate: true, labelMargin: 10,
                tickColor: 'rgba(255, 255, 255, 0)'},
            legend: { show: false },
            shadowSize: 0,
            xaxis: { show: true, tickLength: 0 },
            yaxis: { show: true, tickLength: 0 }
        };
        
        var MILLISECONDS_PER_DAY = (1000 * 60 * 60 * 24);
        
        /**
         * @constructor
         * @type {Style}
         *
         * Provides a linkage to style related aspects taht need to be available
         * from javascript.
         */
        function Style() {
            var calculatedScrollBarWidth = 0;
            
            /*
             * If the browsers supported the flexible box model better, we could handle
             * all of our relative layout using it. Since we don't, we have to do some
             * javascript to get the layout we want. The key issue is being able to
             * take all remaining space in an element. We manage the coordination of this
             * using a "relayout" event on the current hp-page.
             * Some things trigger sending this event are:
             *   opening and closing the activity or help sidebar
             *   opening and closing the master filterbar
             *   resizing the master pane
             *   showing and hiding the details notifications messages
             * Some things that listen for this event are:
             *   details header (DetailsShowView)
             *   details notifications messages
             *   grid panels
             * The event is only triggered if something has actually changed that
             * would cause a re-layout.
             *
             * This should be tied to $normal-animation-duration in the SCSS. 
             */
            this.animationDelay = function () {
                return 510;
            };
            
            this.graphColor = function (index, faded) {
                if (faded) {
                    return GRAPH_COLORS[index % GRAPH_COLORS.length];
                } else {
                    return GRAPH_COLORS_FADED[index % GRAPH_COLORS_FADED.length];
                }
            };
            
            this.graphColors = function () {
                return GRAPH_COLORS.slice(0);
            };
            
            this.plotGuideColor = function () {
                return '#ccc';
            };
            
            this.secondaryBackgroundColor = function () {
                return '#f3f3f3';
            };
            
            this.summaryColor = function (index) {
                return SUMMARY_COLORS[index % SUMMARY_COLORS.length];
            };
            
            this.summaryColors = function () {
                return SUMMARY_COLORS.slice(0);
            };
            
            this.summaryBaseColor = function () {
                return SUMMARY_BASE_COLOR;
            };
            
            this.summarySelectedColor = function () {
                return '#000';
            };
            
            this.statusColor = function (status) {
                return STATUS_COLORS[status];
            };
            
            this.unsetColor = function () {
                return '#d9d9d9';
            };
            
            this.pieNoDataColor = function () {
                return '#fff';
            };
            
            this.dashboardBarGraphOptions = function () {
                return $.extend({}, DASHBOARD_BAR_GRAPH_OPTIONS);
            };
            
            this.dashboardLineGraphOptions = function () {
                return $.extend({}, DASHBOARD_LINE_GRAPH_OPTIONS);
            };
            
            this.map = function () {
                return $.extend({}, MAP);
            };
            
            // deprecated, leave for Fusion's GraphicLogicalSwitch copy for now
            this.related = function () {
                return $.extend({}, MAP);
            };
            
            this.scrollBarWidth = function () {
                if (! calculatedScrollBarWidth) {
                    // http://visualpulse.net/forums/index.php?topic=120.0
                    // Find the Width of the Scrollbar
                    var wide_scroll_html = '<div id="wide_scroll_div_one" ' +
                        'style="width:50px;height:50px;overflow-y:scroll;' +
                        'position:absolute;top:-200px;left:-200px;">' +
                        '<div id="wide_scroll_div_two" ' +
                        'style="height:100px;width:100%"></div></div>'; 
                    // Append our div and add the hmtl to your document for calculations
                    $("body").append(wide_scroll_html);
                    // Getting the width of the surrounding(parent) div
                    // we already know it is 50px since we styled it but just to make sure.
                    var scroll_w1 = $("#wide_scroll_div_one").width();
                    // Find the inner width of the inner(child) div.
                    var scroll_w2 = $("#wide_scroll_div_two").innerWidth(); 
                    calculatedScrollBarWidth = scroll_w1 - scroll_w2; // subtract the difference
                    $("#wide_scroll_div_one").remove(); // remove the html
                }
                return calculatedScrollBarWidth;
            };
            
            this.beautifyTimestamp = function (dateString) {
                var date = Date.parseISOString(dateString);
                var today = new Date();
                today.setHours(0, 0, 0, 0);
                var yesterday = new Date(today.getTime() - MILLISECONDS_PER_DAY);
                var result;
                
                if (date.getTime() > today.getTime()) {
                    result = localizer.getString('core.date.today',
                        [$.format.date(date,
                            localizer.getString('core.date.todayFormat'))]);
                } else if (date.getTime() > yesterday.getTime()) {
                     result = localizer.getString('core.date.yesterday',
                        [$.format.date(date,
                            localizer.getString('core.date.yesterdayFormat'))]);
                } else if (date.getFullYear() >= today.getFullYear()) {
                    result = $.format.date(date,
                        localizer.getString('core.date.thisYearFormat'));
                } else {
                    result = $.format.date(date,
                        localizer.getString('core.date.priorYearFormat'));
                }
                return result;
            };
            
            this.config = function () {
                // hp-double-wide allows Fusion to know whether to structure
                // double wide blades to the higher bay number or lower bay number.
                // Expect to remove once Fusion consumes.
                return $.extend({}, {"hp-double-wide": "high-bay"});
            };
        }

        return new Style();
    }());

    return Style;
});
