// (C) Copyright 2011-2013 Hewlett-Packard Development Company, L.P.
/**
 * @type {DashboardResourcePresenter}
 */
define(['hp/services/IndexService',
    'fs/services/tasks/TasksService',
    'fs/services/alerts/AlertsService',
    'hp/core/EventDispatcher'], 
function(indexService, tasksService, alertsService, EventDispatcher) {"use strict";

    var DashboardResourcePresenter = ( function() {
      
        var ALERT_STATUSES = {
                MAJOR: 'warning',
                OK: 'ok',
                CRITICAL: 'error'
            };
        var TASK_STATUSES = {
                Warning: 'warning',
                Completed: 'ok',
                Error: 'error'
            };
        
        /**
         * @constructor
         */
        function DashboardResourcePresenter() {
          
            var dispatcher = new EventDispatcher();
            var configuration;
            var summary;
            
            function onIndexSuccess(indexResults) {
                if (indexResults && indexResults.members && indexResults.members.length > 0) {
                    summary.name = indexResults.members[0].name;
                    dispatcher.fire("summaryChange", summary);
                }
            }
            
            function onIndexError(error) {
                //console.log(error);
            }
            
            function onTasksSuccess(tasks) {
                $.each(tasks.members, function (index, task) {
                    var status = TASK_STATUSES.hasOwnProperty(task.taskState) ? TASK_STATUSES[task.taskState] : task.taskState;
                    if (summary.hasOwnProperty(status)) {
                        summary[status] += 1;
                    }
                });
                dispatcher.fire("summaryChange", summary);
            }
            
            function onTasksError(error) {
                //console.log(error);
            }
            
            function onAlertsSuccess(alerts) {
                $.each(alerts.members, function (index, alert) {
                    var status = ALERT_STATUSES.hasOwnProperty(alert.severity) ? ALERT_STATUSES[alert.severity] : alert.status;
                    if (summary.hasOwnProperty(status)) {
                        summary[status] += 1;
                    }
                });
                dispatcher.fire("summaryChange", summary);
            }
            
            function onAlertsError(error) {
                //console.log(error);
            }
            
            function getSummary() {
                summary = {name: 'TBD',
                    uri: configuration.resourceUri,
                    error: 0, warning: 0, ok: 0};
                indexService.getIndexForResource(configuration.resourceUri,
                    {success: onIndexSuccess,
                      error: onIndexError});
                tasksService.getTasksByFilter(["associatedResourceUri='" +
                    configuration.resourceUri + "'"], 25,
                    ["created:desc"],
                    {success: onTasksSuccess, error: onTasksError});
                alertsService.getAlertsByFilter(["resourceUri='" +
                    configuration.resourceUri + "'",
                    "alertState='Active'"], false, 25,
                    ["severity:desc", "created:desc"],
                    {success: onAlertsSuccess, error: onAlertsError});
            }
            
            /**
             * @public
             */
            this.init = function (configurationArg) {
                configuration = configurationArg;
            };
            
            this.resume = function () {
            };
            
            this.pause = function () {
            };
            
            this.update = function () {
                getSummary();
            };            
            
            /**
             * Add a listener for a specified event.
             * @public
             * @param {string} eventName The name of the event.
             * @param {function(...)}
             */
            this.on = function(eventName, callback) {
                dispatcher.on(eventName, callback);
            };
            
            this.off = function(eventName, callback) {
                dispatcher.off(eventName, callback);
            };
        }

        return DashboardResourcePresenter;
    }());

    return DashboardResourcePresenter;
});
