// (C) Copyright 2012-2013 Hewlett-Packard Development Company, L.P.
/**
 * @constructor
 * @type {NotificationDetailsView}
 */
define(['hp/core/UrlFragment',
    'hp/core/Localizer',
    'text!hpPages/core/notification_details.html',
    'jquery',
    'hp/lib/jquery.hpTextFormat'],
function(urlFragment, localizer, notificationDetailsHtml) {"use strict";

    var NotificationDetailsView = (function() {
      
        var ROOT = '.hp-notification-details';
        var DETAILS = '.hp-details';
        var RESOLUTION_CONTAINER = '.hp-resolution-container';
        var RESOLUTION = '.hp-resolution';
        var ACTIONS = '.hp-actions';
        var HTML_REGEXP = new RegExp('<\\w.*>');

        /**
         * Constructor
         */
        function NotificationDetailsView() {
            
            function updateTextOrHtml(elem, text) {
                if (HTML_REGEXP.test(text)) {
                    // looks like HTML
                    elem.html(text);
                } else {
                    // text, allow basic formatting
                    elem.hpTextFormat(urlFragment.decodeHtmlEntities(text));
                }
            }
            
            function reset(root) {
                $(DETAILS, root).empty().removeClass('hp-unset');
                $(RESOLUTION, root).empty();
                $(RESOLUTION_CONTAINER, root).hide();
                $(ACTIONS, root).empty();
            }

            function update(root, notif, hideEmpty) {
                var nothing = true;
              
                if (notif.details) {
                    updateTextOrHtml($(DETAILS, root), notif.details);
                    $(DETAILS, root).show();
                    nothing = false;
                }
                if (notif.resolution) {
                    updateTextOrHtml($(RESOLUTION, root), notif.resolution);
                    $(RESOLUTION_CONTAINER, root).show();
                    nothing = false;
                } else {
                    $(RESOLUTION_CONTAINER, root).hide();
                }
                $(ACTIONS, root).empty();
                if (notif.actions && notif.actions.length > 0) {
                    $.each(notif.actions, function (index, action) {
												var item = $('<li></li>');
												updateTextOrHtml(item, action);
                        $(ACTIONS, root).append(item);
                    });
                    $(ACTIONS, root).show();
                    nothing = false;
                } else {
                    $(ACTIONS, root).hide();
                }
                
                if (notif.hasOwnProperty('children')) {
                    var length = notif.children.length;
                    for (var i=0; i<length; i++) {
                        var subRoot = $(notificationDetailsHtml).filter(ROOT);
                        localizer.localizeDom(subRoot);
                        subRoot.addClass('hp-sub-details');
                        root.parent().append(subRoot);
                        if (update(subRoot, notif.children[i], hideEmpty)) {
                            nothing = false;
                        }
                    }
                }
                
                if (nothing) {
                    if (hideEmpty) {
                        $(DETAILS, root).hide();
                    } else {
                        $(DETAILS, root).
                            text(localizer.getString('fs.activity.details.none')).
                            addClass('hp-unset').show();
                    }
                }
                
                // hook to allow formatters to bind DOM events.
                // ugly but needed for Fusion for now
                if (notif.added) {
                    notif.added();
                }
                
                return (! nothing);
            }
            
            function getOrAdd(container) {
                var root = $(ROOT, container);
                if (root.length === 0) {
                    root = $(notificationDetailsHtml).filter(ROOT);
                    localizer.localizeDom(root);
                    $(container).append(root);
                }
                return root;
            }

            /**
             * @public
             * Update the contents of the container with the notification message.
             * @param {container} jQuery selector or object to populate.
             *
             * @param {notification} Object with properties as follows:
             *
             *    status: one of 'ok', 'error', 'warning', 'info'
             *    summary: minimal text summary, should start with an action if possible
             *    changing: true|false, whether the status should be anotated as changing
             *    details: longer description that can have line breaks and links,
             *              will show up in a tooltip/flyout. Will use the hpTextFormat
             *              to format text.
             *    resolution: multi-line recommendation on how the user should proceed
             *    actions: Array of action links in Markdown syntax (preferred) or HTML
             *    children: child notification objects
             *
             *    The only property that is required is the details.
             *
             * @param {hideEmpty} Hide the container if there's nothing to show.
             */
            this.update = function (container, notification, hideEmpty) {
                var root = getOrAdd(container);
                var result = false;
                if (notification) {
                    result = update(root, notification, hideEmpty);
                }
                return result;
            };
            
            this.reset = function (container) {
                var root = getOrAdd(container);
                reset(root);
                $('.hp-sub-details', container).remove();
            };
        }

        return new NotificationDetailsView();
    }());

    return NotificationDetailsView;
});
